#ifndef _DS_LOG_H
#define _DS_LOG_H

#include <glib.h>

typedef void (*DSExceptionHandler)(int err, const gchar *msg);


#define ds_log(log_levels, module, err, ...) \
    _ds_log_message(0, G_LOG_DOMAIN, log_levels,  (log_levels&DS_LOG_FATAL_MESSAGE)?__FILE__:NULL, __func__, __LINE__, NULL, module, NULL, NULL, err, __VA_ARGS__)

#define ds_conditional_log(condition, log_levels, module, err, ...) \
    _ds_log_message(condition, G_LOG_DOMAIN, log_levels,  (log_levels&DS_LOG_FATAL_MESSAGE)?__FILE__:NULL, __func__, __LINE__, NULL, module, NULL, NULL, err, __VA_ARGS__)

#define ds_custom_log(condition, log_levels, module, prefix, err, ...) \
    _ds_log_message(condition, G_LOG_DOMAIN, log_levels, (log_levels&DS_LOG_FATAL_MESSAGE)?__FILE__:NULL, __func__, __LINE__, NULL, module, NULL, prefix, err, __VA_ARGS__)


#define ds_debug(...) ds_log(G_LOG_LEVEL_DEBUG, NULL, 0, __VA_ARGS__)
#define ds_info(...) ds_log(G_LOG_LEVEL_INFO, NULL, 0, __VA_ARGS__)
#define ds_warning(...) ds_log(G_LOG_LEVEL_WARNING, NULL, 0, __VA_ARGS__)
#define ds_error(...) ds_log(G_LOG_LEVEL_CRITICAL, NULL, 0, __VA_ARGS__)
#define ds_fail(...) ds_log((GLogLevelFlags)(G_LOG_LEVEL_ERROR|DS_LOG_FATAL_MESSAGE), NULL, 0, __VA_ARGS__)


#define ds_exception(cond, err, ...) ds_conditional_log(cond, G_LOG_LEVEL_CRITICAL, NULL, err, __VA_ARGS__)
#define ds_exception_reterr(cond, err, ...) if (ds_exception(cond, err, __VA_ARGS__)) return err
#define ds_exception_retnull(cond, err, ...) if (ds_exception(cond, err, __VA_ARGS__)) return NULL

#ifdef __cplusplus
extern "C" {
#endif

/**
  * Pushes message into the glib log file, this is maximal function which is
  * expected to be called from various define macroses
  *
  * @param condition is message is shown only if condition is false
  * @param log_domain is the log domain, usually G_LOG_DOMAIN
  * @param log_level is the log level, either from GLogLevelFlags or a user-defined level, may indicate fatality
  * @param file is source file where the event have occured
  * @param func is function name where the event have occured
  * @param line is source line where the event have occured
  * @param service is the type of service which haev produced the event
  * @param module is the type of module which have produced the event
  * @param ptr is the object which have produced the event
  * @param prefix is a message enclosing err and message (%s replaced with message and %iud... with error code)
  * @param err is error code (0 - not an error, -1 - from errno)
  * @param format is the message format. See the printf() documentation
  */
int _ds_log_message(int condition, const gchar *log_domain, GLogLevelFlags log_levels, const char *file, const char *func, int line, const gchar *service, const gchar *module, void *ptr, gchar *prefix, int err, gchar *format, ...);


#define ds_log_message(condition, log_domain, log_levels, service, module, addr, prefix, err, ...) \
    _ds_log_message(condition, log_domain, log_levels, __FILE__, __func__, __LINE__, service, module, addr, prefix, err, __VA_ARGS__)

/** Terminate application after printing message */
#define DS_LOG_FATAL_MESSAGE G_LOG_FLAG_FATAL
/** Free memory occupied by message if prefix is undefined or prefix otherwise */
#define DS_LOG_CLEAN_MESSAGE G_LOG_FLAG_RECURSION
/** Mask for selecting all levels */
#define DS_LOG_LEVEL_MASK G_LOG_LEVEL_MASK

/** Set required logger output and logging details (for all domains)
  * 
  * @param log_level - maximal log detail
  * @param sink - currently unsupported
  */
void ds_configure_logger(GLogLevelFlags log_level, const char *sink);

/** Set required logging details (for all domains)
  * 
  * @param log_level - maximal log detail
  */
void ds_log_set_level(GLogLevelFlags log_level);

/** Switches on logging message timestmaps */
void ds_log_enable_timestamps();

/** Switches on logging threading information */
void ds_log_enable_threads();

/** Enables specific exception handler */
void ds_log_configure_exception_handler(GLogLevelFlags log_level, DSExceptionHandler handler, int log_exception_message);

# ifdef __cplusplus
}
# endif

#endif /* _DS_LOG_H */
